package org.apache.torque.generated.peer;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.util.List;

import org.apache.torque.BaseDatabaseTestCase;
import org.apache.torque.criteria.Criteria;
import org.apache.torque.test.dbobject.Author;
import org.apache.torque.test.dbobject.Book;
import org.apache.torque.test.peer.AuthorPeer;
import org.apache.torque.test.peer.BookPeer;

/**
 * Tests whether the getRelatedObjects methodd works
 *
 * @version $Id: GetRelatedObjectsTest.java 1395238 2012-10-07 07:30:25Z tfischer $
 */
public class GetRelatedObjectsTest extends BaseDatabaseTestCase
{
    private List<Author> authorList;

    @Override
    public void setUp() throws Exception
    {
        super.setUp();
        cleanBookstore();
        authorList = insertBookstoreData();
    }

    /**
     * Tests getters of related objects.
     *
     * @throws Exception if the test fails
     */
    public void testGetRelatedObjects() throws Exception
    {
        Criteria criteria = new Criteria().where(
                AuthorPeer.AUTHOR_ID,
                authorList.get(0).getAuthorId());
        Author author = AuthorPeer.doSelect(criteria).get(0);

        // execute loading books
        List<Book> books = author.getBooks();

        // verify
        assertEquals(10, books.size());
        assertTrue(books.get(0).getTitle().endsWith("- Author 1"));
    }

    /**
     * Tests getters of related objects, after the collection was initialized.
     *
     * @throws Exception if the test fails
     */
    public void testGetRelatedObjectsAfterInit() throws Exception
    {
        Criteria criteria = new Criteria().where(
                AuthorPeer.AUTHOR_ID,
                authorList.get(0).getAuthorId());
        Author author = AuthorPeer.doSelect(criteria).get(0);

        // execute : init, then load books
        author.initBooks();
        List<Book> books = author.getBooks();

        // verify
        // Books should not be loaded from db but cache should be used
        assertEquals(0, books.size());
    }

    /**
     * Tests getters of related objects with a Criteria object
     *
     * @throws Exception if the test fails
     */
    public void testGetRelatedObjectsWithCriteria() throws Exception
    {
        Criteria criteria = new Criteria().where(
                AuthorPeer.AUTHOR_ID,
                authorList.get(0).getAuthorId());
        Author author = AuthorPeer.doSelect(criteria).get(0);
        Book book = authorList.get(0).getBooks().get(2);

        // execute : load books
        criteria = new Criteria().where(BookPeer.BOOK_ID, book.getBookId());
        List<Book> books = author.getBooks(criteria);

        // verify
        assertEquals(1, books.size());
        assertEquals(book.getBookId(), books.get(0).getBookId());
    }

    /**
     * Tests getters of related objects with a Criteria object
     * after the related objects were loaded with another criteria.
     *
     * @throws Exception if the test fails
     */
    public void testGetRelatedObjectsWithOtherCriteria() throws Exception
    {
        Criteria criteria = new Criteria().where(
                AuthorPeer.AUTHOR_ID,
                authorList.get(0).getAuthorId());
        Author author = AuthorPeer.doSelect(criteria).get(0);
        author.getBooks();
        Book book = authorList.get(0).getBooks().get(2);

        // execute : load books
        criteria = new Criteria().where(BookPeer.BOOK_ID, book.getBookId());
        List<Book> books = author.getBooks(criteria);

        // verify
        assertEquals(1, books.size());
        assertEquals(book.getBookId(), books.get(0).getBookId());
    }

    /**
     * Tests getters of related objects with a Criteria object
     * after the init method was called on the collection.
     *
     * @throws Exception if the test fails
     */
    public void testGetRelatedObjectsWithCriteriaAfterInit() throws Exception
    {
        Criteria criteria = new Criteria().where(
                AuthorPeer.AUTHOR_ID,
                authorList.get(0).getAuthorId());
        Author author = AuthorPeer.doSelect(criteria).get(0);
        author.initBooks();
        Book book = authorList.get(0).getBooks().get(2);

        // execute : load books
        criteria = new Criteria().where(BookPeer.BOOK_ID, book.getBookId());
        List<Book> books = author.getBooks(criteria);

        // verify
        assertEquals(1, books.size());
        assertEquals(book.getBookId(), books.get(0).getBookId());
    }
}
