/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.cluster.loadbalancing;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import org.neo4j.driver.AccessMode;
import org.neo4j.driver.Logger;
import org.neo4j.driver.Logging;
import org.neo4j.driver.exceptions.SecurityException;
import org.neo4j.driver.exceptions.ServiceUnavailableException;
import org.neo4j.driver.exceptions.SessionExpiredException;
import org.neo4j.driver.internal.BoltServerAddress;
import org.neo4j.driver.internal.DomainNameResolver;
import org.neo4j.driver.internal.RoutingErrorHandler;
import org.neo4j.driver.internal.async.ConnectionContext;
import org.neo4j.driver.internal.async.ImmutableConnectionContext;
import org.neo4j.driver.internal.async.connection.RoutingConnection;
import org.neo4j.driver.internal.cluster.Rediscovery;
import org.neo4j.driver.internal.cluster.RediscoveryImpl;
import org.neo4j.driver.internal.cluster.RoutingProcedureClusterCompositionProvider;
import org.neo4j.driver.internal.cluster.RoutingSettings;
import org.neo4j.driver.internal.cluster.RoutingTable;
import org.neo4j.driver.internal.cluster.RoutingTableRegistry;
import org.neo4j.driver.internal.cluster.RoutingTableRegistryImpl;
import org.neo4j.driver.internal.cluster.loadbalancing.LoadBalancingStrategy;
import org.neo4j.driver.internal.messaging.request.MultiDatabaseUtil;
import org.neo4j.driver.internal.shaded.io.netty.util.concurrent.EventExecutorGroup;
import org.neo4j.driver.internal.spi.Connection;
import org.neo4j.driver.internal.spi.ConnectionPool;
import org.neo4j.driver.internal.spi.ConnectionProvider;
import org.neo4j.driver.internal.util.Clock;
import org.neo4j.driver.internal.util.Futures;
import org.neo4j.driver.net.ServerAddressResolver;

public class LoadBalancer
implements ConnectionProvider {
    private static final String CONNECTION_ACQUISITION_COMPLETION_FAILURE_MESSAGE = "Connection acquisition failed for all available addresses.";
    private static final String CONNECTION_ACQUISITION_COMPLETION_EXCEPTION_MESSAGE = "Failed to obtain connection towards %s server. Known routing table is: %s";
    private static final String CONNECTION_ACQUISITION_ATTEMPT_FAILURE_MESSAGE = "Failed to obtain a connection towards address %s, will try other addresses if available. Complete failure is reported separately from this entry.";
    private static final BoltServerAddress[] BOLT_SERVER_ADDRESSES_EMPTY_ARRAY = new BoltServerAddress[0];
    private final ConnectionPool connectionPool;
    private final RoutingTableRegistry routingTables;
    private final LoadBalancingStrategy loadBalancingStrategy;
    private final EventExecutorGroup eventExecutorGroup;
    private final Logger log;
    private final Rediscovery rediscovery;

    public LoadBalancer(BoltServerAddress initialRouter, RoutingSettings settings, ConnectionPool connectionPool, EventExecutorGroup eventExecutorGroup, Clock clock, Logging logging, LoadBalancingStrategy loadBalancingStrategy, ServerAddressResolver resolver, DomainNameResolver domainNameResolver) {
        this(connectionPool, LoadBalancer.createRediscovery(eventExecutorGroup, initialRouter, resolver, settings, clock, logging, Objects.requireNonNull(domainNameResolver)), settings, loadBalancingStrategy, eventExecutorGroup, clock, logging);
    }

    private LoadBalancer(ConnectionPool connectionPool, Rediscovery rediscovery, RoutingSettings settings, LoadBalancingStrategy loadBalancingStrategy, EventExecutorGroup eventExecutorGroup, Clock clock, Logging logging) {
        this(connectionPool, LoadBalancer.createRoutingTables(connectionPool, rediscovery, settings, clock, logging), rediscovery, loadBalancingStrategy, eventExecutorGroup, logging);
    }

    LoadBalancer(ConnectionPool connectionPool, RoutingTableRegistry routingTables, Rediscovery rediscovery, LoadBalancingStrategy loadBalancingStrategy, EventExecutorGroup eventExecutorGroup, Logging logging) {
        this.connectionPool = connectionPool;
        this.routingTables = routingTables;
        this.rediscovery = rediscovery;
        this.loadBalancingStrategy = loadBalancingStrategy;
        this.eventExecutorGroup = eventExecutorGroup;
        this.log = logging.getLog(this.getClass());
    }

    @Override
    public CompletionStage<Connection> acquireConnection(ConnectionContext context) {
        return this.routingTables.ensureRoutingTable(context).thenCompose(handler -> this.acquire(context.mode(), handler.routingTable()).thenApply(connection -> new RoutingConnection((Connection)connection, Futures.joinNowOrElseThrow(context.databaseNameFuture(), ConnectionContext.PENDING_DATABASE_NAME_EXCEPTION_SUPPLIER), context.mode(), context.impersonatedUser(), (RoutingErrorHandler)handler)));
    }

    @Override
    public CompletionStage<Void> verifyConnectivity() {
        return this.supportsMultiDb().thenCompose(supports -> this.routingTables.ensureRoutingTable(ImmutableConnectionContext.simple(supports))).handle((ignored, error) -> {
            if (error != null) {
                Throwable cause = Futures.completionExceptionCause(error);
                if (cause instanceof ServiceUnavailableException) {
                    throw Futures.asCompletionException(new ServiceUnavailableException("Unable to connect to database management service, ensure the database is running and that there is a working network connection to it.", cause));
                }
                throw Futures.asCompletionException(cause);
            }
            return null;
        });
    }

    @Override
    public CompletionStage<Void> close() {
        return this.connectionPool.close();
    }

    @Override
    public CompletionStage<Boolean> supportsMultiDb() {
        List<BoltServerAddress> addresses;
        try {
            addresses = this.rediscovery.resolve();
        }
        catch (Throwable error) {
            return Futures.failedFuture(error);
        }
        CompletableFuture result = Futures.completedWithNull();
        ServiceUnavailableException baseError = new ServiceUnavailableException("Failed to perform multi-databases feature detection with the following servers: " + addresses);
        for (BoltServerAddress address : addresses) {
            result = Futures.onErrorContinue(result, baseError, completionError -> {
                Throwable error = Futures.completionExceptionCause(completionError);
                if (error instanceof SecurityException) {
                    return Futures.failedFuture(error);
                }
                return this.supportsMultiDb(address);
            });
        }
        return Futures.onErrorContinue(result, baseError, completionError -> {
            Throwable error = Futures.completionExceptionCause(completionError);
            if (error instanceof SecurityException) {
                return Futures.failedFuture(error);
            }
            return Futures.failedFuture(baseError);
        });
    }

    public RoutingTableRegistry getRoutingTableRegistry() {
        return this.routingTables;
    }

    private CompletionStage<Boolean> supportsMultiDb(BoltServerAddress address) {
        return this.connectionPool.acquire(address).thenCompose(conn -> {
            boolean supportsMultiDatabase = MultiDatabaseUtil.supportsMultiDatabase(conn);
            return conn.release().thenApply(ignored -> supportsMultiDatabase);
        });
    }

    private CompletionStage<Connection> acquire(AccessMode mode, RoutingTable routingTable) {
        CompletableFuture<Connection> result = new CompletableFuture<Connection>();
        ArrayList<Throwable> attemptExceptions = new ArrayList<Throwable>();
        this.acquire(mode, routingTable, result, attemptExceptions);
        return result;
    }

    private void acquire(AccessMode mode, RoutingTable routingTable, CompletableFuture<Connection> result, List<Throwable> attemptErrors) {
        List<BoltServerAddress> addresses = LoadBalancer.getAddressesByMode(mode, routingTable);
        BoltServerAddress address = this.selectAddress(mode, addresses);
        if (address == null) {
            SessionExpiredException completionError2 = new SessionExpiredException(String.format(CONNECTION_ACQUISITION_COMPLETION_EXCEPTION_MESSAGE, new Object[]{mode, routingTable}));
            attemptErrors.forEach(completionError2::addSuppressed);
            this.log.error(CONNECTION_ACQUISITION_COMPLETION_FAILURE_MESSAGE, completionError2);
            result.completeExceptionally(completionError2);
            return;
        }
        this.connectionPool.acquire(address).whenComplete((connection, completionError) -> {
            Throwable error = Futures.completionExceptionCause(completionError);
            if (error != null) {
                if (error instanceof ServiceUnavailableException) {
                    String attemptMessage = String.format(CONNECTION_ACQUISITION_ATTEMPT_FAILURE_MESSAGE, address);
                    this.log.warn(attemptMessage, new Object[0]);
                    this.log.debug(attemptMessage, error);
                    attemptErrors.add(error);
                    routingTable.forget(address);
                    this.eventExecutorGroup.next().execute(() -> this.acquire(mode, routingTable, result, attemptErrors));
                } else {
                    result.completeExceptionally(error);
                }
            } else {
                result.complete((Connection)connection);
            }
        });
    }

    private static List<BoltServerAddress> getAddressesByMode(AccessMode mode, RoutingTable routingTable) {
        switch (mode) {
            case READ: {
                return routingTable.readers();
            }
            case WRITE: {
                return routingTable.writers();
            }
        }
        throw LoadBalancer.unknownMode(mode);
    }

    private BoltServerAddress selectAddress(AccessMode mode, List<BoltServerAddress> addresses) {
        switch (mode) {
            case READ: {
                return this.loadBalancingStrategy.selectReader(addresses);
            }
            case WRITE: {
                return this.loadBalancingStrategy.selectWriter(addresses);
            }
        }
        throw LoadBalancer.unknownMode(mode);
    }

    private static RoutingTableRegistry createRoutingTables(ConnectionPool connectionPool, Rediscovery rediscovery, RoutingSettings settings, Clock clock, Logging logging) {
        return new RoutingTableRegistryImpl(connectionPool, rediscovery, clock, logging, settings.routingTablePurgeDelayMs());
    }

    private static Rediscovery createRediscovery(EventExecutorGroup eventExecutorGroup, BoltServerAddress initialRouter, ServerAddressResolver resolver, RoutingSettings settings, Clock clock, Logging logging, DomainNameResolver domainNameResolver) {
        RoutingProcedureClusterCompositionProvider clusterCompositionProvider = new RoutingProcedureClusterCompositionProvider(clock, settings.routingContext(), logging);
        return new RediscoveryImpl(initialRouter, settings, clusterCompositionProvider, eventExecutorGroup, resolver, logging, domainNameResolver);
    }

    private static RuntimeException unknownMode(AccessMode mode) {
        return new IllegalArgumentException("Mode '" + (Object)((Object)mode) + "' is not supported");
    }
}

