/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.sling.jcr.base.internal.mount;

import java.util.Set;
import javax.jcr.AccessDeniedException;
import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.UnsupportedRepositoryOperationException;
import javax.jcr.Workspace;

import org.apache.jackrabbit.api.JackrabbitSession;
import org.apache.jackrabbit.api.JackrabbitWorkspace;
import org.apache.jackrabbit.api.security.principal.PrincipalManager;
import org.apache.jackrabbit.api.security.user.UserManager;

public class ProxyJackrabbitSession extends ProxySession<JackrabbitSession> implements JackrabbitSession {
    public ProxyJackrabbitSession(ProxyRepository repository, JackrabbitSession jcr, Session mount, Set<String> mountPoints) {
        super(repository, jcr, mount, mountPoints);
    }

    @Override
    public Workspace getWorkspace() {
        return new ProxyJackrabbitWorkspace(this, (JackrabbitWorkspace) this.jcr.getWorkspace(), (JackrabbitWorkspace) this.mount.getWorkspace());
    }

    public boolean hasPermission(String absPath, String... actions) throws RepositoryException {
        if (isMount(absPath)) {
            return ((JackrabbitSession) mount).hasPermission(absPath, actions);
        }
        return jcr.hasPermission(absPath, actions);
    }

    public PrincipalManager getPrincipalManager() throws AccessDeniedException, UnsupportedRepositoryOperationException, RepositoryException {
        return jcr.getPrincipalManager();
    }

    public UserManager getUserManager() throws AccessDeniedException, UnsupportedRepositoryOperationException, RepositoryException {
        return new ProxyUserManager(this, jcr.getUserManager(), ((JackrabbitSession) mount).getUserManager());
    }

    public Item getItemOrNull(String absPath) throws RepositoryException {
        if (super.itemExists(absPath)) {
            return super.getItem(absPath);
        } else {
            return null;
        }
    }

    public Property getPropertyOrNull(String absPath) throws RepositoryException {
        if (super.propertyExists(absPath)) {
            return super.getProperty(absPath);
        } else {
            return null;
        }
    }

    public Node getNodeOrNull(String absPath) throws RepositoryException {
        if (super.nodeExists(absPath)) {
            return super.getNode(absPath);
        } else {
            return null;
        }
    }
}
