/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.repair.consistent;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.AsyncFunction;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.repair.RepairSessionResult;
import org.apache.cassandra.repair.SomeRepairFailedException;
import org.apache.cassandra.repair.consistent.ConsistentSession;
import org.apache.cassandra.repair.messages.FailSession;
import org.apache.cassandra.repair.messages.FinalizeCommit;
import org.apache.cassandra.repair.messages.FinalizePropose;
import org.apache.cassandra.repair.messages.PrepareConsistentRequest;
import org.apache.cassandra.repair.messages.RepairMessage;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CoordinatorSession
extends ConsistentSession {
    private static final Logger logger = LoggerFactory.getLogger(CoordinatorSession.class);
    private final Map<InetAddressAndPort, ConsistentSession.State> participantStates = new HashMap<InetAddressAndPort, ConsistentSession.State>();
    private final SettableFuture<Boolean> prepareFuture = SettableFuture.create();
    private final SettableFuture<Boolean> finalizeProposeFuture = SettableFuture.create();
    private volatile long sessionStart = Long.MIN_VALUE;
    private volatile long repairStart = Long.MIN_VALUE;
    private volatile long finalizeStart = Long.MIN_VALUE;

    public CoordinatorSession(Builder builder) {
        super(builder);
        for (InetAddressAndPort participant : this.participants) {
            this.participantStates.put(participant, ConsistentSession.State.PREPARING);
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public void setState(ConsistentSession.State state) {
        logger.trace("Setting coordinator state to {} for repair {}", (Object)state, (Object)this.sessionID);
        super.setState(state);
    }

    @VisibleForTesting
    synchronized ConsistentSession.State getParticipantState(InetAddressAndPort participant) {
        return this.participantStates.get(participant);
    }

    public synchronized void setParticipantState(InetAddressAndPort participant, ConsistentSession.State state) {
        logger.trace("Setting participant {} to state {} for repair {}", new Object[]{participant, state, this.sessionID});
        Preconditions.checkArgument((boolean)this.participantStates.containsKey(participant), (String)"Session %s doesn't include %s", (Object)this.sessionID, (Object)participant);
        Preconditions.checkArgument((boolean)this.participantStates.get(participant).canTransitionTo(state), (String)"Invalid state transition %s -> %s", (Object)((Object)this.participantStates.get(participant)), (Object)((Object)state));
        this.participantStates.put(participant, state);
        if (Iterables.all(this.participantStates.values(), s -> s == state)) {
            this.setState(state);
        }
    }

    synchronized void setAll(ConsistentSession.State state) {
        for (InetAddressAndPort participant : this.participants) {
            this.setParticipantState(participant, state);
        }
    }

    synchronized boolean allStates(ConsistentSession.State state) {
        return this.getState() == state && Iterables.all(this.participantStates.values(), v -> v == state);
    }

    synchronized boolean hasFailed() {
        return this.getState() == ConsistentSession.State.FAILED || Iterables.any(this.participantStates.values(), v -> v == ConsistentSession.State.FAILED);
    }

    protected void sendMessage(InetAddressAndPort destination, Message<RepairMessage> message) {
        logger.trace("Sending {} to {}", message.payload, (Object)destination);
        MessagingService.instance().send(message, destination);
    }

    public ListenableFuture<Boolean> prepare() {
        Preconditions.checkArgument((boolean)this.allStates(ConsistentSession.State.PREPARING));
        logger.info("Beginning prepare phase of incremental repair session {}", (Object)this.sessionID);
        Message<RepairMessage> message = Message.out(Verb.PREPARE_CONSISTENT_REQ, new PrepareConsistentRequest(this.sessionID, this.coordinator, (Set<InetAddressAndPort>)this.participants));
        for (InetAddressAndPort participant : this.participants) {
            this.sendMessage(participant, message);
        }
        return this.prepareFuture;
    }

    public synchronized void handlePrepareResponse(InetAddressAndPort participant, boolean success) {
        if (this.getState() == ConsistentSession.State.FAILED) {
            logger.trace("Incremental repair {} has failed, ignoring prepare response from {}", (Object)this.sessionID, (Object)participant);
            return;
        }
        if (!success) {
            logger.warn("{} failed the prepare phase for incremental repair session {}", (Object)participant, (Object)this.sessionID);
            this.sendFailureMessageToParticipants();
            this.setParticipantState(participant, ConsistentSession.State.FAILED);
        } else {
            logger.trace("Successful prepare response received from {} for repair session {}", (Object)participant, (Object)this.sessionID);
            this.setParticipantState(participant, ConsistentSession.State.PREPARED);
        }
        if (Iterables.any(this.participantStates.values(), v -> v == ConsistentSession.State.PREPARING)) {
            return;
        }
        if (this.getState() == ConsistentSession.State.PREPARED) {
            logger.info("Incremental repair session {} successfully prepared.", (Object)this.sessionID);
            this.prepareFuture.set((Object)true);
        } else {
            this.fail();
            this.prepareFuture.set((Object)false);
        }
    }

    public synchronized void setRepairing() {
        this.setAll(ConsistentSession.State.REPAIRING);
    }

    public synchronized ListenableFuture<Boolean> finalizePropose() {
        Preconditions.checkArgument((boolean)this.allStates(ConsistentSession.State.REPAIRING));
        logger.info("Proposing finalization of repair session {}", (Object)this.sessionID);
        Message<RepairMessage> message = Message.out(Verb.FINALIZE_PROPOSE_MSG, new FinalizePropose(this.sessionID));
        for (InetAddressAndPort participant : this.participants) {
            this.sendMessage(participant, message);
        }
        return this.finalizeProposeFuture;
    }

    public synchronized void handleFinalizePromise(InetAddressAndPort participant, boolean success) {
        if (this.getState() == ConsistentSession.State.FAILED) {
            logger.trace("Incremental repair {} has failed, ignoring finalize promise from {}", (Object)this.sessionID, (Object)participant);
        } else if (!success) {
            logger.warn("Finalization proposal of session {} rejected by {}. Aborting session", (Object)this.sessionID, (Object)participant);
            this.fail();
            this.finalizeProposeFuture.set((Object)false);
        } else {
            logger.trace("Successful finalize promise received from {} for repair session {}", (Object)participant, (Object)this.sessionID);
            this.setParticipantState(participant, ConsistentSession.State.FINALIZE_PROMISED);
            if (this.getState() == ConsistentSession.State.FINALIZE_PROMISED) {
                logger.info("Finalization proposal for repair session {} accepted by all participants.", (Object)this.sessionID);
                this.finalizeProposeFuture.set((Object)true);
            }
        }
    }

    public synchronized void finalizeCommit() {
        Preconditions.checkArgument((boolean)this.allStates(ConsistentSession.State.FINALIZE_PROMISED));
        logger.info("Committing finalization of repair session {}", (Object)this.sessionID);
        Message<RepairMessage> message = Message.out(Verb.FINALIZE_COMMIT_MSG, new FinalizeCommit(this.sessionID));
        for (InetAddressAndPort participant : this.participants) {
            this.sendMessage(participant, message);
        }
        this.setAll(ConsistentSession.State.FINALIZED);
        logger.info("Incremental repair session {} completed", (Object)this.sessionID);
    }

    private void sendFailureMessageToParticipants() {
        Message<RepairMessage> message = Message.out(Verb.FAILED_SESSION_MSG, new FailSession(this.sessionID));
        for (InetAddressAndPort participant : this.participants) {
            if (this.participantStates.get(participant) == ConsistentSession.State.FAILED) continue;
            this.sendMessage(participant, message);
        }
    }

    public synchronized void fail() {
        Set cantFail = this.participantStates.entrySet().stream().filter(entry -> !((ConsistentSession.State)((Object)((Object)entry.getValue()))).canTransitionTo(ConsistentSession.State.FAILED)).collect(Collectors.toSet());
        if (!cantFail.isEmpty()) {
            logger.error("Can't transition endpoints {} to FAILED", cantFail, (Object)new RuntimeException());
            return;
        }
        logger.info("Incremental repair session {} failed", (Object)this.sessionID);
        this.sendFailureMessageToParticipants();
        this.setAll(ConsistentSession.State.FAILED);
        String exceptionMsg = String.format("Incremental repair session %s has failed", this.sessionID);
        this.finalizeProposeFuture.setException((Throwable)new RuntimeException(exceptionMsg));
        this.prepareFuture.setException((Throwable)new RuntimeException(exceptionMsg));
    }

    private static String formatDuration(long then, long now) {
        if (then == Long.MIN_VALUE || now == Long.MIN_VALUE) {
            return "n/a";
        }
        return DurationFormatUtils.formatDurationWords((long)(now - then), (boolean)true, (boolean)true);
    }

    public ListenableFuture execute(final Supplier<ListenableFuture<List<RepairSessionResult>>> sessionSubmitter, final AtomicBoolean hasFailure) {
        logger.info("Beginning coordination of incremental repair session {}", (Object)this.sessionID);
        this.sessionStart = System.currentTimeMillis();
        ListenableFuture<Boolean> prepareResult = this.prepare();
        ListenableFuture repairSessionResults = Futures.transformAsync(prepareResult, (AsyncFunction)new AsyncFunction<Boolean, List<RepairSessionResult>>(){

            public ListenableFuture<List<RepairSessionResult>> apply(Boolean success) throws Exception {
                if (success.booleanValue()) {
                    CoordinatorSession.this.repairStart = System.currentTimeMillis();
                    if (logger.isDebugEnabled()) {
                        logger.debug("Incremental repair {} prepare phase completed in {}", (Object)CoordinatorSession.this.sessionID, (Object)CoordinatorSession.formatDuration(CoordinatorSession.this.sessionStart, CoordinatorSession.this.repairStart));
                    }
                    CoordinatorSession.this.setRepairing();
                    return (ListenableFuture)sessionSubmitter.get();
                }
                return Futures.immediateFuture(null);
            }
        }, (Executor)MoreExecutors.directExecutor());
        ListenableFuture proposeFuture = Futures.transformAsync((ListenableFuture)repairSessionResults, (AsyncFunction)new AsyncFunction<List<RepairSessionResult>, Boolean>(){

            public ListenableFuture<Boolean> apply(List<RepairSessionResult> results) throws Exception {
                if (results == null || results.isEmpty() || Iterables.any(results, r -> r == null)) {
                    CoordinatorSession.this.finalizeStart = System.currentTimeMillis();
                    if (logger.isDebugEnabled()) {
                        logger.debug("Incremental repair {} validation/stream phase completed in {}", (Object)CoordinatorSession.this.sessionID, (Object)CoordinatorSession.formatDuration(CoordinatorSession.this.repairStart, CoordinatorSession.this.finalizeStart));
                    }
                    return Futures.immediateFailedFuture((Throwable)SomeRepairFailedException.INSTANCE);
                }
                return CoordinatorSession.this.finalizePropose();
            }
        }, (Executor)MoreExecutors.directExecutor());
        final SettableFuture resultFuture = SettableFuture.create();
        Futures.addCallback((ListenableFuture)proposeFuture, (FutureCallback)new FutureCallback<Boolean>(){

            public void onSuccess(@Nullable Boolean result) {
                try {
                    if (result != null && result.booleanValue()) {
                        if (logger.isDebugEnabled()) {
                            logger.debug("Incremental repair {} finalization phase completed in {}", (Object)CoordinatorSession.this.sessionID, (Object)CoordinatorSession.formatDuration(CoordinatorSession.this.finalizeStart, System.currentTimeMillis()));
                        }
                        CoordinatorSession.this.finalizeCommit();
                        if (logger.isDebugEnabled()) {
                            logger.debug("Incremental repair {} phase completed in {}", (Object)CoordinatorSession.this.sessionID, (Object)CoordinatorSession.formatDuration(CoordinatorSession.this.sessionStart, System.currentTimeMillis()));
                        }
                    } else {
                        hasFailure.set(true);
                        CoordinatorSession.this.fail();
                    }
                    resultFuture.set((Object)result);
                }
                catch (Exception e) {
                    resultFuture.setException((Throwable)e);
                }
            }

            public void onFailure(Throwable t) {
                try {
                    if (logger.isDebugEnabled()) {
                        logger.debug("Incremental repair {} phase failed in {}", (Object)CoordinatorSession.this.sessionID, (Object)CoordinatorSession.formatDuration(CoordinatorSession.this.sessionStart, System.currentTimeMillis()));
                    }
                    hasFailure.set(true);
                    CoordinatorSession.this.fail();
                }
                finally {
                    resultFuture.setException(t);
                }
            }
        }, (Executor)MoreExecutors.directExecutor());
        return resultFuture;
    }

    public static class Builder
    extends ConsistentSession.AbstractBuilder {
        public CoordinatorSession build() {
            this.validate();
            return new CoordinatorSession(this);
        }
    }
}

