/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.async;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiFunction;
import java.util.function.Function;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Query;
import org.neo4j.driver.TransactionConfig;
import org.neo4j.driver.async.ResultCursor;
import org.neo4j.driver.exceptions.AuthorizationExpiredException;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.exceptions.ConnectionReadTimeoutException;
import org.neo4j.driver.internal.BookmarkHolder;
import org.neo4j.driver.internal.async.ResultCursorsHolder;
import org.neo4j.driver.internal.cursor.AsyncResultCursor;
import org.neo4j.driver.internal.cursor.RxResultCursor;
import org.neo4j.driver.internal.messaging.BoltProtocol;
import org.neo4j.driver.internal.spi.Connection;
import org.neo4j.driver.internal.util.Futures;
import org.neo4j.driver.internal.util.LockUtil;

public class UnmanagedTransaction {
    protected static final String CANT_COMMIT_COMMITTED_MSG = "Can't commit, transaction has been committed";
    protected static final String CANT_ROLLBACK_COMMITTED_MSG = "Can't rollback, transaction has been committed";
    protected static final String CANT_COMMIT_ROLLED_BACK_MSG = "Can't commit, transaction has been rolled back";
    protected static final String CANT_ROLLBACK_ROLLED_BACK_MSG = "Can't rollback, transaction has been rolled back";
    protected static final String CANT_COMMIT_ROLLING_BACK_MSG = "Can't commit, transaction has been requested to be rolled back";
    protected static final String CANT_ROLLBACK_COMMITTING_MSG = "Can't rollback, transaction has been requested to be committed";
    private static final EnumSet<State> OPEN_STATES = EnumSet.of(State.ACTIVE, State.TERMINATED);
    private final Connection connection;
    private final BoltProtocol protocol;
    private final BookmarkHolder bookmarkHolder;
    private final ResultCursorsHolder resultCursors;
    private final long fetchSize;
    private final Lock lock = new ReentrantLock();
    private State state = State.ACTIVE;
    private CompletableFuture<Void> commitFuture;
    private CompletableFuture<Void> rollbackFuture;
    private Throwable causeOfTermination;

    public UnmanagedTransaction(Connection connection, BookmarkHolder bookmarkHolder, long fetchSize) {
        this(connection, bookmarkHolder, fetchSize, new ResultCursorsHolder());
    }

    protected UnmanagedTransaction(Connection connection, BookmarkHolder bookmarkHolder, long fetchSize, ResultCursorsHolder resultCursors) {
        this.connection = connection;
        this.protocol = connection.protocol();
        this.bookmarkHolder = bookmarkHolder;
        this.resultCursors = resultCursors;
        this.fetchSize = fetchSize;
    }

    public CompletionStage<UnmanagedTransaction> beginAsync(Bookmark initialBookmark, TransactionConfig config) {
        return this.protocol.beginTransaction(this.connection, initialBookmark, config).handle((ignore, beginError) -> {
            if (beginError != null) {
                if (beginError instanceof AuthorizationExpiredException) {
                    this.connection.terminateAndRelease("Authorization information kept on the server has expired, this connection is no longer valid.");
                } else if (beginError instanceof ConnectionReadTimeoutException) {
                    this.connection.terminateAndRelease(beginError.getMessage());
                } else {
                    this.connection.release();
                }
                throw Futures.asCompletionException(beginError);
            }
            return this;
        });
    }

    public CompletionStage<Void> closeAsync() {
        return this.closeAsync(false);
    }

    public CompletionStage<Void> closeAsync(boolean commit) {
        return this.closeAsync(commit, true);
    }

    public CompletionStage<Void> commitAsync() {
        return this.closeAsync(true, false);
    }

    public CompletionStage<Void> rollbackAsync() {
        return this.closeAsync(false, false);
    }

    public CompletionStage<ResultCursor> runAsync(Query query) {
        this.ensureCanRunQueries();
        CompletionStage<AsyncResultCursor> cursorStage = this.protocol.runInUnmanagedTransaction(this.connection, query, this, this.fetchSize).asyncResult();
        this.resultCursors.add(cursorStage);
        return cursorStage.thenCompose(AsyncResultCursor::mapSuccessfulRunCompletionAsync).thenApply(cursor -> cursor);
    }

    public CompletionStage<RxResultCursor> runRx(Query query) {
        this.ensureCanRunQueries();
        CompletionStage<RxResultCursor> cursorStage = this.protocol.runInUnmanagedTransaction(this.connection, query, this, this.fetchSize).rxResult();
        this.resultCursors.add(cursorStage);
        return cursorStage;
    }

    public boolean isOpen() {
        return OPEN_STATES.contains((Object)LockUtil.executeWithLock(this.lock, () -> this.state));
    }

    public void markTerminated(Throwable cause) {
        LockUtil.executeWithLock(this.lock, () -> {
            if (this.state == State.TERMINATED) {
                if (this.causeOfTermination != null) {
                    this.addSuppressedWhenNotCaptured(this.causeOfTermination, cause);
                }
            } else {
                this.state = State.TERMINATED;
                this.causeOfTermination = cause;
            }
        });
    }

    private void addSuppressedWhenNotCaptured(Throwable currentCause, Throwable newCause) {
        boolean noneMatch;
        if (currentCause != newCause && (noneMatch = Arrays.stream(currentCause.getSuppressed()).noneMatch(suppressed -> suppressed == newCause))) {
            currentCause.addSuppressed(newCause);
        }
    }

    public Connection connection() {
        return this.connection;
    }

    private void ensureCanRunQueries() {
        LockUtil.executeWithLock(this.lock, () -> {
            if (this.state == State.COMMITTED) {
                throw new ClientException("Cannot run more queries in this transaction, it has been committed");
            }
            if (this.state == State.ROLLED_BACK) {
                throw new ClientException("Cannot run more queries in this transaction, it has been rolled back");
            }
            if (this.state == State.TERMINATED) {
                throw new ClientException("Cannot run more queries in this transaction, it has either experienced an fatal error or was explicitly terminated", this.causeOfTermination);
            }
        });
    }

    private CompletionStage<Void> doCommitAsync(Throwable cursorFailure) {
        ClientException exception = LockUtil.executeWithLock(this.lock, () -> this.state == State.TERMINATED ? new ClientException("Transaction can't be committed. It has been rolled back either because of an error or explicit termination", cursorFailure != this.causeOfTermination ? this.causeOfTermination : null) : null);
        return exception != null ? Futures.failedFuture(exception) : this.protocol.commitTransaction(this.connection).thenAccept(this.bookmarkHolder::setBookmark);
    }

    private CompletionStage<Void> doRollbackAsync() {
        return LockUtil.executeWithLock(this.lock, () -> this.state) == State.TERMINATED ? Futures.completedWithNull() : this.protocol.rollbackTransaction(this.connection);
    }

    private static BiFunction<Void, Throwable, Void> handleCommitOrRollback(Throwable cursorFailure) {
        return (ignore, commitOrRollbackError) -> {
            CompletionException combinedError = Futures.combineErrors(cursorFailure, commitOrRollbackError);
            if (combinedError != null) {
                throw combinedError;
            }
            return null;
        };
    }

    private void handleTransactionCompletion(boolean commitAttempt, Throwable throwable) {
        LockUtil.executeWithLock(this.lock, () -> {
            this.state = commitAttempt && throwable == null ? State.COMMITTED : State.ROLLED_BACK;
        });
        if (throwable instanceof AuthorizationExpiredException) {
            this.connection.terminateAndRelease("Authorization information kept on the server has expired, this connection is no longer valid.");
        } else if (throwable instanceof ConnectionReadTimeoutException) {
            this.connection.terminateAndRelease(throwable.getMessage());
        } else {
            this.connection.release();
        }
    }

    private CompletionStage<Void> closeAsync(boolean commit, boolean completeWithNullIfNotOpen) {
        CompletableFuture<Void> stage = LockUtil.executeWithLock(this.lock, () -> {
            CompletableFuture<Object> resultStage = null;
            if (completeWithNullIfNotOpen && !this.isOpen()) {
                resultStage = Futures.completedWithNull();
            } else if (this.state == State.COMMITTED) {
                resultStage = Futures.failedFuture(new ClientException(commit ? CANT_COMMIT_COMMITTED_MSG : CANT_ROLLBACK_COMMITTED_MSG));
            } else if (this.state == State.ROLLED_BACK) {
                resultStage = Futures.failedFuture(new ClientException(commit ? CANT_COMMIT_ROLLED_BACK_MSG : CANT_ROLLBACK_ROLLED_BACK_MSG));
            } else if (commit) {
                if (this.rollbackFuture != null) {
                    resultStage = Futures.failedFuture(new ClientException(CANT_COMMIT_ROLLING_BACK_MSG));
                } else if (this.commitFuture != null) {
                    resultStage = this.commitFuture;
                } else {
                    this.commitFuture = new CompletableFuture();
                }
            } else if (this.commitFuture != null) {
                resultStage = Futures.failedFuture(new ClientException(CANT_ROLLBACK_COMMITTING_MSG));
            } else if (this.rollbackFuture != null) {
                resultStage = this.rollbackFuture;
            } else {
                this.rollbackFuture = new CompletableFuture();
            }
            return resultStage;
        });
        if (stage == null) {
            Function<Throwable, CompletionStage> targetAction;
            CompletableFuture<Void> targetFuture;
            if (commit) {
                targetFuture = this.commitFuture;
                targetAction = throwable -> this.doCommitAsync((Throwable)throwable).handle(UnmanagedTransaction.handleCommitOrRollback(throwable));
            } else {
                targetFuture = this.rollbackFuture;
                targetAction = throwable -> this.doRollbackAsync().handle(UnmanagedTransaction.handleCommitOrRollback(throwable));
            }
            this.resultCursors.retrieveNotConsumedError().thenCompose(targetAction).whenComplete((ignored, throwable) -> this.handleTransactionCompletion(commit, (Throwable)throwable)).whenComplete(Futures.futureCompletingConsumer(targetFuture));
            stage = targetFuture;
        }
        return stage;
    }

    private static enum State {
        ACTIVE,
        TERMINATED,
        COMMITTED,
        ROLLED_BACK;

    }
}

