/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionController;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategy;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategyOptions;
import org.apache.cassandra.db.compaction.TimeWindowCompactionStrategyOptions;
import org.apache.cassandra.db.compaction.TimeWindowCompactionTask;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.db.lifecycle.SSTableSet;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.schema.CompactionParams;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimeWindowCompactionStrategy
extends AbstractCompactionStrategy {
    private static final Logger logger = LoggerFactory.getLogger(TimeWindowCompactionStrategy.class);
    private final TimeWindowCompactionStrategyOptions options;
    protected volatile int estimatedRemainingTasks = 0;
    private final Set<SSTableReader> sstables = new HashSet<SSTableReader>();
    private long lastExpiredCheck;
    private long highestWindowSeen;

    public TimeWindowCompactionStrategy(ColumnFamilyStore cfs, Map<String, String> options) {
        super(cfs, options);
        this.options = new TimeWindowCompactionStrategyOptions(options);
        String[] tsOpts = new String[]{"unchecked_tombstone_compaction", "tombstone_compaction_interval", "tombstone_threshold"};
        if (Arrays.stream(tsOpts).map(o -> (String)options.get(o)).filter(Objects::nonNull).anyMatch(v -> !v.equals("false"))) {
            logger.debug("Enabling tombstone compactions for TWCS");
        } else {
            logger.debug("Disabling tombstone compactions for TWCS");
            this.disableTombstoneCompactions = true;
        }
    }

    @Override
    public AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        List<SSTableReader> previousCandidate = null;
        List<SSTableReader> latestBucket;
        while (!(latestBucket = this.getNextBackgroundSSTables(gcBefore)).isEmpty()) {
            if (latestBucket.equals(previousCandidate)) {
                logger.warn("Could not acquire references for compacting SSTables {} which is not a problem per se,unless it happens frequently, in which case it must be reported. Will retry later.", latestBucket);
                return null;
            }
            LifecycleTransaction modifier = this.cfs.getTracker().tryModify(latestBucket, OperationType.COMPACTION);
            if (modifier != null) {
                return new TimeWindowCompactionTask(this.cfs, modifier, gcBefore, this.options.ignoreOverlaps);
            }
            previousCandidate = latestBucket;
        }
        return null;
    }

    private synchronized List<SSTableReader> getNextBackgroundSSTables(int gcBefore) {
        if (Iterables.isEmpty(this.cfs.getSSTables(SSTableSet.LIVE))) {
            return Collections.emptyList();
        }
        ImmutableSet uncompacting = ImmutableSet.copyOf((Iterable)Iterables.filter(this.cfs.getUncompactingSSTables(), this.sstables::contains));
        Set<Object> expired = Collections.emptySet();
        if (Clock.Global.currentTimeMillis() - this.lastExpiredCheck > this.options.expiredSSTableCheckFrequency) {
            logger.debug("TWCS expired check sufficiently far in the past, checking for fully expired SSTables");
            expired = CompactionController.getFullyExpiredSSTables(this.cfs, (Iterable<SSTableReader>)uncompacting, this.options.ignoreOverlaps ? Collections.emptySet() : this.cfs.getOverlappingLiveSSTables((Iterable<SSTableReader>)uncompacting), gcBefore, this.options.ignoreOverlaps);
            this.lastExpiredCheck = Clock.Global.currentTimeMillis();
        } else {
            logger.debug("TWCS skipping check for fully expired SSTables");
        }
        HashSet candidates = Sets.newHashSet(TimeWindowCompactionStrategy.filterSuspectSSTables((Iterable<SSTableReader>)uncompacting));
        ArrayList<SSTableReader> compactionCandidates = new ArrayList<SSTableReader>(this.getNextNonExpiredSSTables((Iterable<SSTableReader>)Sets.difference((Set)candidates, expired), gcBefore));
        if (!expired.isEmpty()) {
            logger.debug("Including expired sstables: {}", expired);
            compactionCandidates.addAll(expired);
        }
        return compactionCandidates;
    }

    private List<SSTableReader> getNextNonExpiredSSTables(Iterable<SSTableReader> nonExpiringSSTables, int gcBefore) {
        List<SSTableReader> mostInteresting = this.getCompactionCandidates(nonExpiringSSTables);
        if (mostInteresting != null) {
            return mostInteresting;
        }
        ArrayList<SSTableReader> sstablesWithTombstones = new ArrayList<SSTableReader>();
        for (SSTableReader sstable : nonExpiringSSTables) {
            if (!this.worthDroppingTombstones(sstable, gcBefore)) continue;
            sstablesWithTombstones.add(sstable);
        }
        if (sstablesWithTombstones.isEmpty()) {
            return Collections.emptyList();
        }
        return Collections.singletonList(Collections.min(sstablesWithTombstones, SSTableReader.sizeComparator));
    }

    private List<SSTableReader> getCompactionCandidates(Iterable<SSTableReader> candidateSSTables) {
        Pair<HashMultimap<Long, SSTableReader>, Long> buckets = TimeWindowCompactionStrategy.getBuckets(candidateSSTables, this.options.sstableWindowUnit, this.options.sstableWindowSize, this.options.timestampResolution);
        if ((Long)buckets.right > this.highestWindowSeen) {
            this.highestWindowSeen = (Long)buckets.right;
        }
        NewestBucket mostInteresting = TimeWindowCompactionStrategy.newestBucket((HashMultimap<Long, SSTableReader>)((HashMultimap)buckets.left), this.cfs.getMinimumCompactionThreshold(), this.cfs.getMaximumCompactionThreshold(), this.options.stcsOptions, this.highestWindowSeen);
        this.estimatedRemainingTasks = mostInteresting.estimatedRemainingTasks;
        if (!mostInteresting.sstables.isEmpty()) {
            return mostInteresting.sstables;
        }
        return null;
    }

    @Override
    public synchronized void addSSTable(SSTableReader sstable) {
        this.sstables.add(sstable);
    }

    @Override
    public synchronized void removeSSTable(SSTableReader sstable) {
        this.sstables.remove(sstable);
    }

    @Override
    protected synchronized Set<SSTableReader> getSSTables() {
        return ImmutableSet.copyOf(this.sstables);
    }

    public static Pair<Long, Long> getWindowBoundsInMillis(TimeUnit windowTimeUnit, int windowTimeSize, long timestampInMillis) {
        long upperTimestamp;
        long lowerTimestamp;
        long timestampInSeconds = TimeUnit.SECONDS.convert(timestampInMillis, TimeUnit.MILLISECONDS);
        switch (windowTimeUnit) {
            case MINUTES: {
                lowerTimestamp = timestampInSeconds - timestampInSeconds % (60L * (long)windowTimeSize);
                upperTimestamp = lowerTimestamp + 60L * ((long)windowTimeSize - 1L) + 59L;
                break;
            }
            case HOURS: {
                lowerTimestamp = timestampInSeconds - timestampInSeconds % (3600L * (long)windowTimeSize);
                upperTimestamp = lowerTimestamp + 3600L * ((long)windowTimeSize - 1L) + 3599L;
                break;
            }
            default: {
                lowerTimestamp = timestampInSeconds - timestampInSeconds % (86400L * (long)windowTimeSize);
                upperTimestamp = lowerTimestamp + 86400L * ((long)windowTimeSize - 1L) + 86399L;
            }
        }
        return Pair.create(TimeUnit.MILLISECONDS.convert(lowerTimestamp, TimeUnit.SECONDS), TimeUnit.MILLISECONDS.convert(upperTimestamp, TimeUnit.SECONDS));
    }

    @VisibleForTesting
    static Pair<HashMultimap<Long, SSTableReader>, Long> getBuckets(Iterable<SSTableReader> files, TimeUnit sstableWindowUnit, int sstableWindowSize, TimeUnit timestampResolution) {
        HashMultimap buckets = HashMultimap.create();
        long maxTimestamp = 0L;
        for (SSTableReader f : files) {
            assert (TimeWindowCompactionStrategyOptions.validTimestampTimeUnits.contains((Object)timestampResolution));
            long tStamp = TimeUnit.MILLISECONDS.convert(f.getMaxTimestamp(), timestampResolution);
            Pair<Long, Long> bounds = TimeWindowCompactionStrategy.getWindowBoundsInMillis(sstableWindowUnit, sstableWindowSize, tStamp);
            buckets.put(bounds.left, (Object)f);
            if ((Long)bounds.left <= maxTimestamp) continue;
            maxTimestamp = (Long)bounds.left;
        }
        logger.trace("buckets {}, max timestamp {}", (Object)buckets, (Object)maxTimestamp);
        return Pair.create(buckets, maxTimestamp);
    }

    @VisibleForTesting
    static NewestBucket newestBucket(HashMultimap<Long, SSTableReader> buckets, int minThreshold, int maxThreshold, SizeTieredCompactionStrategyOptions stcsOptions, long now) {
        List<SSTableReader> sstables = Collections.emptyList();
        int estimatedRemainingTasks = 0;
        TreeSet allKeys = new TreeSet(buckets.keySet());
        Iterator it = allKeys.descendingIterator();
        while (it.hasNext()) {
            Long key = (Long)it.next();
            Set bucket = buckets.get((Object)key);
            logger.trace("Key {}, now {}", (Object)key, (Object)now);
            if (bucket.size() >= minThreshold && key >= now) {
                List pairs = SizeTieredCompactionStrategy.createSSTableAndLengthPairs(bucket);
                List<List<SSTableReader>> stcsBuckets = SizeTieredCompactionStrategy.getBuckets(pairs, stcsOptions.bucketHigh, stcsOptions.bucketLow, stcsOptions.minSSTableSize);
                List<SSTableReader> stcsInterestingBucket = SizeTieredCompactionStrategy.mostInterestingBucket(stcsBuckets, minThreshold, maxThreshold);
                if (stcsInterestingBucket.isEmpty()) continue;
                double remaining = bucket.size() - maxThreshold;
                estimatedRemainingTasks = (int)((double)estimatedRemainingTasks + (1.0 + (remaining > (double)minThreshold ? Math.ceil(remaining / (double)maxThreshold) : 0.0)));
                if (sstables.isEmpty()) {
                    logger.debug("Using STCS compaction for first window of bucket: data files {} , options {}", pairs, (Object)stcsOptions);
                    sstables = stcsInterestingBucket;
                    continue;
                }
                logger.trace("First window of bucket is eligible but not selected: data files {} , options {}", pairs, (Object)stcsOptions);
                continue;
            }
            if (bucket.size() >= 2 && key < now) {
                double remaining = bucket.size() - maxThreshold;
                estimatedRemainingTasks = (int)((double)estimatedRemainingTasks + (1.0 + (remaining > (double)minThreshold ? Math.ceil(remaining / (double)maxThreshold) : 0.0)));
                if (sstables.isEmpty()) {
                    logger.debug("bucket size {} >= 2 and not in current bucket, compacting what's here: {}", (Object)bucket.size(), (Object)bucket);
                    sstables = TimeWindowCompactionStrategy.trimToThreshold(bucket, maxThreshold);
                    continue;
                }
                logger.trace("bucket size {} >= 2 and not in current bucket, eligible but not selected: {}", (Object)bucket.size(), (Object)bucket);
                continue;
            }
            logger.trace("No compaction necessary for bucket size {} , key {}, now {}", new Object[]{bucket.size(), key, now});
        }
        return new NewestBucket(sstables, estimatedRemainingTasks);
    }

    @VisibleForTesting
    static List<SSTableReader> trimToThreshold(Set<SSTableReader> bucket, int maxThreshold) {
        ArrayList<SSTableReader> ssTableReaders = new ArrayList<SSTableReader>(bucket);
        Collections.sort(ssTableReaders, SSTableReader.sizeComparator);
        return ImmutableList.copyOf((Iterable)Iterables.limit(ssTableReaders, (int)maxThreshold));
    }

    @Override
    public synchronized Collection<AbstractCompactionTask> getMaximalTask(int gcBefore, boolean splitOutput) {
        List<SSTableReader> filteredSSTables = TimeWindowCompactionStrategy.filterSuspectSSTables(this.sstables);
        if (Iterables.isEmpty(filteredSSTables)) {
            return null;
        }
        LifecycleTransaction txn = this.cfs.getTracker().tryModify(filteredSSTables, OperationType.COMPACTION);
        if (txn == null) {
            return null;
        }
        return Collections.singleton(new TimeWindowCompactionTask(this.cfs, txn, gcBefore, this.options.ignoreOverlaps));
    }

    @Override
    public Collection<Collection<SSTableReader>> groupSSTablesForAntiCompaction(Collection<SSTableReader> sstablesToGroup) {
        ArrayList<Collection<SSTableReader>> groups = new ArrayList<Collection<SSTableReader>>(sstablesToGroup.size());
        for (SSTableReader sstable : sstablesToGroup) {
            groups.add(Collections.singleton(sstable));
        }
        return groups;
    }

    @Override
    public synchronized AbstractCompactionTask getUserDefinedTask(Collection<SSTableReader> sstables, int gcBefore) {
        assert (!sstables.isEmpty());
        LifecycleTransaction modifier = this.cfs.getTracker().tryModify(sstables, OperationType.COMPACTION);
        if (modifier == null) {
            logger.debug("Unable to mark {} for compaction; probably a background compaction got to it first.  You can disable background compactions temporarily if this is a problem", sstables);
            return null;
        }
        return new TimeWindowCompactionTask(this.cfs, modifier, gcBefore, this.options.ignoreOverlaps).setUserDefined(true);
    }

    @Override
    public int getEstimatedRemainingTasks() {
        return this.estimatedRemainingTasks;
    }

    @Override
    public long getMaxSSTableBytes() {
        return Long.MAX_VALUE;
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        Map<String, String> uncheckedOptions = AbstractCompactionStrategy.validateOptions(options);
        uncheckedOptions = TimeWindowCompactionStrategyOptions.validateOptions(options, uncheckedOptions);
        uncheckedOptions.remove(CompactionParams.Option.MIN_THRESHOLD.toString());
        uncheckedOptions.remove(CompactionParams.Option.MAX_THRESHOLD.toString());
        return uncheckedOptions;
    }

    public String toString() {
        return String.format("TimeWindowCompactionStrategy[%s/%s]", this.cfs.getMinimumCompactionThreshold(), this.cfs.getMaximumCompactionThreshold());
    }

    static final class NewestBucket {
        final List<SSTableReader> sstables;
        final int estimatedRemainingTasks;

        NewestBucket(List<SSTableReader> sstables, int estimatedRemainingTasks) {
            this.sstables = sstables;
            this.estimatedRemainingTasks = estimatedRemainingTasks;
        }

        public String toString() {
            return String.format("sstables: %s, estimated remaining tasks: %d", this.sstables, this.estimatedRemainingTasks);
        }
    }
}

