/*
 * Decompiled with CFR 0.152.
 */
package com.azure.storage.common;

import com.azure.core.http.HttpPipeline;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.util.Context;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.logging.ClientLogger;
import com.azure.storage.common.implementation.StorageImplUtils;
import com.azure.storage.common.policy.StorageSharedKeyCredentialPolicy;
import java.net.URL;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

public final class StorageSharedKeyCredential {
    private final ClientLogger logger = new ClientLogger(StorageSharedKeyCredential.class);
    private static final String AUTHORIZATION_HEADER_FORMAT = "SharedKey %s:%s";
    private static final Context LOG_STRING_TO_SIGN_CONTEXT = new Context((Object)"Azure-Storage-Log-String-To-Sign", (Object)true);
    private static final String ACCOUNT_NAME = "accountname";
    private static final String ACCOUNT_KEY = "accountkey";
    private final String accountName;
    private final String accountKey;

    public StorageSharedKeyCredential(String accountName, String accountKey) {
        Objects.requireNonNull(accountName, "'accountName' cannot be null.");
        Objects.requireNonNull(accountKey, "'accountKey' cannot be null.");
        this.accountName = accountName;
        this.accountKey = accountKey;
    }

    public static StorageSharedKeyCredential fromConnectionString(String connectionString) {
        HashMap<String, String> connectionStringPieces = new HashMap<String, String>();
        for (String connectionStringPiece : connectionString.split(";")) {
            String[] kvp = connectionStringPiece.split("=", 2);
            connectionStringPieces.put(kvp[0].toLowerCase(Locale.ROOT), kvp[1]);
        }
        String accountName = (String)connectionStringPieces.get(ACCOUNT_NAME);
        String accountKey = (String)connectionStringPieces.get(ACCOUNT_KEY);
        if (CoreUtils.isNullOrEmpty((CharSequence)accountName) || CoreUtils.isNullOrEmpty((CharSequence)accountKey)) {
            throw new IllegalArgumentException("Connection string must contain 'AccountName' and 'AccountKey'.");
        }
        return new StorageSharedKeyCredential(accountName, accountKey);
    }

    public String getAccountName() {
        return this.accountName;
    }

    public String generateAuthorizationHeader(URL requestURL, String httpMethod, Map<String, String> headers) {
        return this.generateAuthorizationHeader(requestURL, httpMethod, headers, false);
    }

    public String generateAuthorizationHeader(URL requestURL, String httpMethod, Map<String, String> headers, boolean logStringToSign) {
        String signature = StorageImplUtils.computeHMac256(this.accountKey, this.buildStringToSign(requestURL, httpMethod, headers, logStringToSign));
        return String.format(AUTHORIZATION_HEADER_FORMAT, this.accountName, signature);
    }

    public String computeHmac256(String stringToSign) {
        return StorageImplUtils.computeHMac256(this.accountKey, stringToSign);
    }

    private String buildStringToSign(URL requestURL, String httpMethod, Map<String, String> headers, boolean logStringToSign) {
        String contentLength = headers.get("Content-Length");
        contentLength = contentLength.equals("0") ? "" : contentLength;
        String dateHeader = headers.containsKey("x-ms-date") ? "" : this.getStandardHeaderValue(headers, "Date");
        String stringToSign = String.join((CharSequence)"\n", httpMethod, this.getStandardHeaderValue(headers, "Content-Encoding"), this.getStandardHeaderValue(headers, "Content-Language"), contentLength, this.getStandardHeaderValue(headers, "Content-MD5"), this.getStandardHeaderValue(headers, "Content-Type"), dateHeader, this.getStandardHeaderValue(headers, "If-Modified-Since"), this.getStandardHeaderValue(headers, "If-Match"), this.getStandardHeaderValue(headers, "If-None-Match"), this.getStandardHeaderValue(headers, "If-Unmodified-Since"), this.getStandardHeaderValue(headers, "Range"), this.getAdditionalXmsHeaders(headers), this.getCanonicalizedResource(requestURL));
        if (logStringToSign) {
            StorageImplUtils.logStringToSign(this.logger, stringToSign, LOG_STRING_TO_SIGN_CONTEXT);
        }
        return stringToSign;
    }

    private String getStandardHeaderValue(Map<String, String> headers, String headerName) {
        String headerValue = headers.get(headerName);
        return headerValue == null ? "" : headerValue;
    }

    private String getAdditionalXmsHeaders(Map<String, String> headers) {
        List xmsHeaderNameArray = headers.entrySet().stream().filter(entry -> ((String)entry.getKey()).toLowerCase(Locale.ROOT).startsWith("x-ms-")).filter(entry -> entry.getValue() != null).map(Map.Entry::getKey).collect(Collectors.toList());
        if (xmsHeaderNameArray.isEmpty()) {
            return "";
        }
        Collections.sort(xmsHeaderNameArray, Collator.getInstance(Locale.ROOT));
        StringBuilder canonicalizedHeaders = new StringBuilder();
        for (String key : xmsHeaderNameArray) {
            if (canonicalizedHeaders.length() > 0) {
                canonicalizedHeaders.append('\n');
            }
            canonicalizedHeaders.append(key.toLowerCase(Locale.ROOT)).append(':').append(headers.get(key));
        }
        return canonicalizedHeaders.toString();
    }

    private String getCanonicalizedResource(URL requestURL) {
        StringBuilder canonicalizedResource = new StringBuilder("/");
        canonicalizedResource.append(this.accountName);
        if (requestURL.getPath().length() > 0) {
            canonicalizedResource.append(requestURL.getPath());
        } else {
            canonicalizedResource.append('/');
        }
        if (requestURL.getQuery() == null) {
            return canonicalizedResource.toString();
        }
        Map<String, String[]> queryParams = StorageImplUtils.parseQueryStringSplitValues(requestURL.getQuery());
        ArrayList<String> queryParamNames = new ArrayList<String>(queryParams.keySet());
        Collections.sort(queryParamNames);
        for (String queryParamName : queryParamNames) {
            Object[] queryParamValues = queryParams.get(queryParamName);
            Arrays.sort(queryParamValues);
            String queryParamValuesStr = String.join((CharSequence)",", (CharSequence[])queryParamValues);
            canonicalizedResource.append("\n").append(queryParamName.toLowerCase(Locale.ROOT)).append(":").append(queryParamValuesStr);
        }
        return canonicalizedResource.toString();
    }

    public static StorageSharedKeyCredential getSharedKeyCredentialFromPipeline(HttpPipeline httpPipeline) {
        for (int i = 0; i < httpPipeline.getPolicyCount(); ++i) {
            HttpPipelinePolicy httpPipelinePolicy = httpPipeline.getPolicy(i);
            if (!(httpPipelinePolicy instanceof StorageSharedKeyCredentialPolicy)) continue;
            StorageSharedKeyCredentialPolicy storageSharedKeyCredentialPolicy = (StorageSharedKeyCredentialPolicy)httpPipelinePolicy;
            return storageSharedKeyCredentialPolicy.sharedKeyCredential();
        }
        return null;
    }
}

