/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.query.control;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.iotdb.db.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.query.control.clientsession.ClientSession;
import org.apache.iotdb.db.query.control.clientsession.IClientSession;
import org.apache.iotdb.db.service.basic.ServiceProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionTimeoutManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(SessionTimeoutManager.class);
    private static final long MINIMUM_CLEANUP_PERIOD = 2000L;
    private static final long SESSION_TIMEOUT = IoTDBDescriptor.getInstance().getConfig().getSessionTimeoutThreshold();
    private Map<IClientSession, Long> sessionToLastActiveTime;
    private ScheduledExecutorService executorService;

    private SessionTimeoutManager() {
        if (SESSION_TIMEOUT == 0L) {
            return;
        }
        this.sessionToLastActiveTime = new ConcurrentHashMap<IClientSession, Long>();
        this.executorService = IoTDBThreadPoolFactory.newScheduledThreadPool(1, "session-timeout-manager");
        this.executorService.scheduleAtFixedRate(() -> {
            LOGGER.info("cleaning up expired sessions");
            this.cleanup();
        }, 0L, Math.max(2000L, SESSION_TIMEOUT / 5L), TimeUnit.MILLISECONDS);
    }

    public void register(IClientSession session) {
        if (SESSION_TIMEOUT == 0L) {
            return;
        }
        this.sessionToLastActiveTime.put(session, System.currentTimeMillis());
    }

    public boolean unregister(IClientSession session) {
        if (SESSION_TIMEOUT == 0L) {
            return ServiceProvider.SESSION_MANAGER.releaseSessionResource(session);
        }
        if (ServiceProvider.SESSION_MANAGER.releaseSessionResource(session)) {
            return this.sessionToLastActiveTime.remove(session) != null;
        }
        return false;
    }

    public void refresh(IClientSession session) {
        if (SESSION_TIMEOUT == 0L) {
            return;
        }
        this.sessionToLastActiveTime.computeIfPresent(session, (k, v) -> System.currentTimeMillis());
    }

    private void cleanup() {
        long currentTime = System.currentTimeMillis();
        this.sessionToLastActiveTime.entrySet().stream().filter(entry -> (Long)entry.getValue() + SESSION_TIMEOUT < currentTime).forEach(entry -> {
            if (this.unregister((IClientSession)entry.getKey())) {
                LOGGER.debug(String.format("session-%s timed out in %d ms", entry.getKey(), currentTime - (Long)entry.getValue()));
                if (entry.getKey() instanceof ClientSession) {
                    ((ClientSession)entry.getKey()).shutdownStream();
                }
            }
        });
    }

    public static SessionTimeoutManager getInstance() {
        return SessionTimeoutManagerHelper.INSTANCE;
    }

    public boolean isSessionAlive(IClientSession session) {
        if (SESSION_TIMEOUT == 0L) {
            return true;
        }
        return this.sessionToLastActiveTime.containsKey(session);
    }

    private static class SessionTimeoutManagerHelper {
        private static final SessionTimeoutManager INSTANCE = new SessionTimeoutManager();

        private SessionTimeoutManagerHelper() {
        }
    }
}

