"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ts = require("typescript");
function validateDiagnostics(diagnostics, strict) {
    // Print error diagnostics.
    const checkDiagnostics = (diagnostics) => {
        if (diagnostics && diagnostics.length > 0) {
            let errors = '';
            errors = errors + '\n' + ts.formatDiagnostics(diagnostics, {
                getCurrentDirectory: () => ts.sys.getCurrentDirectory(),
                getNewLine: () => ts.sys.newLine,
                getCanonicalFileName: (f) => f,
            });
            return errors;
        }
    };
    const hasError = diagnostics.some(diag => diag.category === ts.DiagnosticCategory.Error);
    if (hasError) {
        // Throw only if we're in strict mode, otherwise return original content.
        if (strict) {
            throw new Error(`
        TS failed with the following error messages:

        ${checkDiagnostics(diagnostics)}
      `);
        }
        else {
            return false;
        }
    }
    return true;
}
function transformJavascript(options) {
    const { content, getTransforms, emitSourceMap, inputFilePath, outputFilePath, strict, } = options;
    // Bail if there's no transform to do.
    if (getTransforms.length === 0) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    const allowFastPath = options.typeCheck === false && !emitSourceMap;
    const outputs = new Map();
    const tempFilename = 'bo-default-file.js';
    const tempSourceFile = ts.createSourceFile(tempFilename, content, ts.ScriptTarget.Latest, allowFastPath);
    const parseDiagnostics = tempSourceFile.parseDiagnostics;
    const tsOptions = {
        // We target latest so that there is no downleveling.
        target: ts.ScriptTarget.Latest,
        isolatedModules: true,
        suppressOutputPathCheck: true,
        allowNonTsExtensions: true,
        noLib: true,
        noResolve: true,
        sourceMap: emitSourceMap,
        inlineSources: emitSourceMap,
        inlineSourceMap: false,
    };
    if (allowFastPath && parseDiagnostics) {
        if (!validateDiagnostics(parseDiagnostics, strict)) {
            return {
                content: null,
                sourceMap: null,
                emitSkipped: true,
            };
        }
        const transforms = getTransforms.map((getTf) => getTf(undefined));
        const result = ts.transform(tempSourceFile, transforms, tsOptions);
        if (result.transformed.length === 0 || result.transformed[0] === tempSourceFile) {
            return {
                content: null,
                sourceMap: null,
                emitSkipped: true,
            };
        }
        const printer = ts.createPrinter(undefined, {
            onEmitNode: result.emitNodeWithNotification,
            substituteNode: result.substituteNode,
        });
        const output = printer.printFile(result.transformed[0]);
        result.dispose();
        return {
            content: output,
            sourceMap: null,
            emitSkipped: false,
        };
    }
    const host = {
        getSourceFile: (fileName) => {
            if (fileName !== tempFilename) {
                throw new Error(`File ${fileName} does not have a sourceFile.`);
            }
            return tempSourceFile;
        },
        getDefaultLibFileName: () => 'lib.d.ts',
        getCurrentDirectory: () => '',
        getDirectories: () => [],
        getCanonicalFileName: (fileName) => fileName,
        useCaseSensitiveFileNames: () => true,
        getNewLine: () => '\n',
        fileExists: (fileName) => fileName === tempFilename,
        readFile: (_fileName) => '',
        writeFile: (fileName, text) => outputs.set(fileName, text),
    };
    const program = ts.createProgram([tempFilename], tsOptions, host);
    const diagnostics = program.getSyntacticDiagnostics(tempSourceFile);
    if (!validateDiagnostics(diagnostics, strict)) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    // We need the checker inside transforms.
    const transforms = getTransforms.map((getTf) => getTf(program));
    program.emit(undefined, undefined, undefined, undefined, { before: transforms, after: [] });
    let transformedContent = outputs.get(tempFilename);
    if (!transformedContent) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    let sourceMap = null;
    const tsSourceMap = outputs.get(`${tempFilename}.map`);
    if (emitSourceMap && tsSourceMap) {
        const urlRegExp = /^\/\/# sourceMappingURL=[^\r\n]*/gm;
        sourceMap = JSON.parse(tsSourceMap);
        // Fix sourcemaps file references.
        if (outputFilePath) {
            sourceMap.file = outputFilePath;
            transformedContent = transformedContent.replace(urlRegExp, `//# sourceMappingURL=${sourceMap.file}.map\n`);
            if (inputFilePath) {
                sourceMap.sources = [inputFilePath];
            }
            else {
                sourceMap.sources = [''];
            }
        }
        else {
            // TODO: figure out if we should inline sources here.
            transformedContent = transformedContent.replace(urlRegExp, '');
            sourceMap.file = '';
            sourceMap.sources = [''];
        }
    }
    return {
        content: transformedContent,
        sourceMap,
        emitSkipped: false,
    };
}
exports.transformJavascript = transformJavascript;
//# sourceMappingURL=data:application/json;base64,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