/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.servicecomb.swagger.generator.core.utils;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import io.swagger.v3.oas.annotations.Operation;


public class MethodUtils {
  /**
   * Get the methods of <code>cls</code> which are valid for generating Swagger schema.
   * @param cls The REST interface class, or so called "controller" class, to be analysed.
   * @return the valid methods to be used to generate Swagger schema, sorted by their Swagger operation name.
   */
  public static List<Method> findSwaggerMethods(Class<?> cls) {
    Method[] methods = cls.getMethods();
    List<Method> result = new ArrayList<>(methods.length);

    for (Method m : methods) {
      if (!isSkipMethod(cls, m)) {
        result.add(m);
      }
    }

    // order of cls.getMethods() is undefined and not stable
    // so we must sort them first to make generation is stable
    result.sort(Comparator.comparing(MethodUtils::findSwaggerMethodName));
    return result;
  }

  public static Map<String, Method> findSwaggerMethodsMapOfOperationId(Class<?> cls) {
    List<Method> methods = findSwaggerMethods(cls);
    Map<String, Method> result = new HashMap<>();
    methods.forEach((item) -> result.put(findSwaggerMethodName(item), item));
    return result;
  }

  /**
   * Pick out those methods not proper to be added into the Swagger schema.
   *
   * @param cls the owner class of the <code>method</code>
   * @param method the method to be validate
   * @return true if this method should be abandoned;
   * false if this method should be added in to Swagger schema
   */
  public static boolean isSkipMethod(Class<?> cls, Method method) {
    if (method.isDefault()) {
      return true;
    }
    if (method.getDeclaringClass() == Object.class) {
      return true;
    }
    if (method.getDeclaringClass().isInterface()
        && !cls.isInterface()) {
      // inherited template methods
      return true;
    }
    // skip static method
    int modifiers = method.getModifiers();
    if (Modifier.isStatic(modifiers)) {
      return true;
    }
    // skip bridge method
    if (method.isBridge()) {
      return true;
    }

    Operation apiOperation = method.getAnnotation(Operation.class);
    return apiOperation != null && apiOperation.hidden();
  }

  /**
   * Get the operationId in schema of this method,
   * no matter whether it should be hidden.
   * @return If the operation name is specified via {@link Operation}, use that one.
   * Otherwise the method name is returned.
   */
  public static String findSwaggerMethodName(Method method) {
    Operation apiOperationAnnotation = method.getAnnotation(Operation.class);
    if (apiOperationAnnotation == null || StringUtils.isEmpty(apiOperationAnnotation.operationId())) {
      return method.getName();
    }

    return apiOperationAnnotation.operationId();
  }
}
